﻿<#

.SYNOPSIS

CLOUDIWAY                                      
© Copyright 2023 Cloudiway, Inc. All Rights Reserved.

.DESCRIPTION
    Script to generate an assessment of an Microsoft 365 tenant and calculate exactly the number of Cloudiway licenses to
    migrate its content to a destination tenant (Microsoft 365 or Google Workspace) using Cloudiway's migration platform
	
.NOTES
	Author			Pablo Galan Sabugo <pgalan@cloudiway.com> 
	Date		    Jan/2023
	Disclaimer: 	This script is provided 'AS IS'. No warrantee is provided either expressed or implied. 
                    Cloudiway cannot be held responsible for any misuse of the script.
    Version: 1.1
#>

import-Module ".\CloudiwayLibrary\CloudiwayLibrary.dll"  -Global  -DisableNameChecking -Force
#Import-Module ".\CloudiwayFunctions\CloudiwayFunctions.psm1" -DisableNameChecking -Force

#######################################################################################################################
#                                               CHECK POWERSHELL MODULES
#######################################################################################################################

Function Check-PSModules {
    param 
    (      
        [Parameter(Mandatory = $true)]
        [string]$logFile
    )

        if (!( ($null -ne (Get-Module -Name "Microsoft.Graph")) -or ($null -ne (Get-InstalledModule -Name "Microsoft.Graph" -ErrorAction SilentlyContinue) ) ) ) {

        Log-Write -LogFile $logFile "WARNING: Microsoft.Graph module not installed." -ForegroundColor Yellow  
        Log-Write -LogFile $logFile "Installing Microsoft.Graph module: " -NoNewLine $true

        sleep 5
    
        try {
            Install-Module -Name Microsoft.Graph -Repository PSGallery -SkipPublisherCheck -AllowClobber -Force -ErrorAction Stop

            Log-Write -LogFile $logFile "SUCCESS. Installed Version: $((Get-InstalledModule Microsoft.Graph).version)" -ForegroundColor "Green"
        }
        catch {
            Log-Write -LogFile $logFile "ERROR: Failed to install Microsoft.Graph module. Script will abort." -ForegroundColor Red
            Log-Write -LogFile $logFile "Run this script 'As administrator' to install the Microsoft.Graph module." -ForegroundColor Yellow
            Exit
        }
    }    
    else {
        Log-Write -LogFile $logFile "Microsoft.Graph module: " -NoNewLine $true
        try {
            Get-InstalledModule Microsoft.Graph > $null
        }
        catch {
            Install-Module -Name Microsoft.Graph -Repository PSGallery  -SkipPublisherCheck -AllowClobber -Force -ErrorAction Stop
          
        }
        Update-Module Microsoft.Graph
        Log-Write -LogFile $logFile "SUCCESS. Installed Version: $((Get-InstalledModule Microsoft.Graph).version)" -ForegroundColor "Green"
           
    }   


    if (!( ($null -ne (Get-Module -Name "ExchangeOnlineManagement")) -or ($null -ne (Get-InstalledModule -Name "ExchangeOnlineManagement" -ErrorAction SilentlyContinue) )) -or !((Get-InstalledModule -Name "ExchangeOnlineManagement").Version -ge 3 ) ) {
       
        Log-Write -LogFile $logFile "WARNING: ExchangeOnlineManagement module not installed." -ForegroundColor Yellow            
        Log-Write -LogFile $logFile "Installing ExchangeOnlineManagement module: " -NoNewLine $true 

        sleep 5
    
        try {
            Install-Module -Name ExchangeOnlineManagement -Repository PSGallery -AllowClobber -Force -ErrorAction Stop
            Log-Write -LogFile $logFile "SUCCESS." -ForegroundColor "Green"
        }
        catch {
            Log-Write -LogFile $logFile "ERROR: Failed to install ExchangeOnlineManagement module. Script will abort." -ForegroundColor Red 
            Log-Write -LogFile $logFile "Run this script 'As administrator' to install the ExchangeOnlineManagement module." -ForegroundColor Yellow
            Exit
        }
        Import-Module ExchangeOnlineManagement -Global
    }
    else {
        Log-Write -LogFile $logFile "ExchangeOnlineManagement module: " -NoNewLine $true
        Log-Write -LogFile $logFile "SUCCESS." -ForegroundColor "Green"
    }

    if (!( ($null -ne (Get-Module -Name "MSOnline")) -or ($null -ne (Get-InstalledModule -Name "MSOnline" -ErrorAction SilentlyContinue) ) ) ) {
       
        Log-Write -LogFile $logFile "WARNING: MSOnline module not installed." -ForegroundColor Yellow            
        Log-Write -LogFile $logFile "Installing MSOnline module: " -NoNewLine $true

        sleep 5
    
        try {
            Install-Module -Name MSOnline -Repository PSGallery -AllowClobber -Force -ErrorAction Stop
            Log-Write -LogFile $logFile "SUCCESS." -ForegroundColor "Green"
        }
        catch {
            Log-Write -LogFile $logFile "ERROR: Failed to install MSOnline module. Script will abort." -ForegroundColor Red 
            Log-Write -LogFile $logFile "Run this script 'As administrator' to install the MSOnline module." -ForegroundColor Yellow
            Exit
        }
        Import-Module MSOnline -Scope Global  #-UseWindowsPowershell
    }
    else {
        Log-Write -LogFile $logFile "MSOnline module: " -NoNewLine $true
        Import-Module MSOnline -Scope Global  #-UseWindowsPowershell
        Log-Write -LogFile $logFile "SUCCESS." -ForegroundColor "Green"             
    }

    if (!( ($null -ne (Get-Module -Name "AzureAD")) -or ($null -ne (Get-InstalledModule -Name "AzureAD" -ErrorAction SilentlyContinue) ) ) ) {

        Log-Write -LogFile $logFile "WARNING: AzureAD module not installed." -ForegroundColor Yellow 
        Log-Write -LogFile $logFile "Installing AzureAD module: " -NoNewLine $true

        sleep 5
    
        try {
            Install-Module -Name AzureAD -Repository PSGallery -AllowClobber -Force -ErrorAction Stop
            Log-Write -LogFile $logFile "SUCCESS." -ForegroundColor "Green"
        }
        catch {
            Log-Write -LogFile $logFile "ERROR: Failed to install AzureAD module. Script will abort." -ForegroundColor Red
            Log-Write -LogFile $logFile "Run this script 'As administrator' to install the AzureAD module." -ForegroundColor Yellow 
            Exit
        }
        Import-Module AzureAD  -Scope Global  #-UseWindowsPowershell 
    }
    else {
        Log-Write -LogFile $logFile "AzureAD module: " -NoNewLine $true
        Import-Module AzureAD  -Scope Global  #-UseWindowsPowershell 
        Log-Write -LogFile $logFile "SUCCESS." -ForegroundColor "Green"     
    } 
    
    if (!( ($null -ne (Get-Module -Name "MicrosoftTeams")) -or ($null -ne (Get-InstalledModule -Name "MicrosoftTeams" -ErrorAction SilentlyContinue) ) ) ) {

        Log-Write -LogFile $logFile "WARNING: MicrosoftTeams module not installed." -ForegroundColor Yellow 
        Log-Write -LogFile $logFile "Installing MicrosoftTeams module: " -NoNewLine $true

        sleep 5
    
        try {
            Install-Module -Name MicrosoftTeams -Repository PSGallery -AllowClobber -Force -ErrorAction Stop
            Log-Write -LogFile $logFile "SUCCESS." -ForegroundColor "Green"
        }
        catch {
            Log-Write -LogFile $logFile "ERROR: Failed to install MicrosoftTeams module. Script will abort." -ForegroundColor Red 
            Log-Write -LogFile $logFile "Run this script 'As administrator' to install the MicrosoftTeams module."  -ForegroundColor Yellow
            Exit
        }
        Import-Module MicrosoftTeams -Scope Global 
    }
    else {
        Log-Write -LogFile $logFile "MicrosoftTeams module: " -NoNewLine $true
        Import-Module MicrosoftTeams -Scope Global 
        Log-Write -LogFile $logFile "SUCCESS." -ForegroundColor "Green"
    }
    
    if (!( ($null -ne (Get-Module -Name "Microsoft.Online.SharePoint.PowerShell")) -or ($null -ne (Get-InstalledModule -Name "Microsoft.Online.SharePoint.PowerShell" -ErrorAction SilentlyContinue) ) ) ) {

        Log-Write -LogFile $logFile "WARNING: Microsoft SharePoint Online Services module not installed." -ForegroundColor Yellow  
        Log-Write -LogFile $logFile "Installing Microsoft SharePoint Online Services module: " -NoNewLine $true

        sleep 5
    
        try {
            Install-Module -Name Microsoft.Online.SharePoint.PowerShell -Repository PSGallery -AllowClobber -Force -ErrorAction Stop
            Log-Write -LogFile $logFile "SUCCESS." -ForegroundColor "Green"
        }
        catch {
            Log-Write -LogFile $logFile "ERROR: Failed to install Microsoft SharePoint Online Services module. Script will abort." -ForegroundColor Red 
            Log-Write -LogFile $logFile "Run this script 'As administrator' to install the Microsoft SharePoint Online Services module." -ForegroundColor Yellow 
            Exit
        }
        Import-Module Microsoft.Online.SharePoint.PowerShell  -DisableNameChecking -Scope Global  #-UseWindowsPowershell 


    }
    else {
        Log-Write -LogFile $logFile "Microsoft.Online.SharePoint.PowerShell module: " -NoNewLine $true
        Import-Module Microsoft.Online.SharePoint.PowerShell  -DisableNameChecking -Scope Global  #-UseWindowsPowershell 
        Log-Write -LogFile $logFile "SUCCESS." -ForegroundColor "Green" 
    }
    
    if (!( ($null -ne (Get-Module -Name "ImportExcel")) -or ($null -ne (Get-InstalledModule -Name "ImportExcel" -ErrorAction SilentlyContinue) ) ) ) {

        Log-Write -LogFile $logFile "WARNING: ImportExcel module not installed." -ForegroundColor Yellow  
        Log-Write -LogFile $logFile "Installing ImportExcel module: " -NoNewLine $true

        sleep 5
    
        try {
            Install-Module -Name "ImportExcel" -Repository PSGallery -AllowClobber -Force -ErrorAction Stop
            Log-Write -LogFile $logFile "SUCCESS." -ForegroundColor "Green"
        }
        catch {
            Log-Write -LogFile $logFile "ERROR: Failed to install ImportExcel module. Script will abort." -ForegroundColor Red
            Log-Write -LogFile $logFile "Run this script 'As administrator' to install the ImportExcel module." -ForegroundColor Yellow
            Exit
        }
        Import-Module "ImportExcel" -Scope Global 
    }
    else {
        Log-Write -LogFile $logFile "ImportExcel module: " -NoNewLine $true
        Log-Write -LogFile $logFile "SUCCESS." -ForegroundColor "Green"
           
    } 
    
    if (!( ($null -ne (Get-Module -Name "JWTDetails")) -or ($null -ne (Get-InstalledModule -Name "JWTDetails" -ErrorAction SilentlyContinue) ) ) ) {

        Log-Write -LogFile $logFile "WARNING: JWTDetails module not installed." -ForegroundColor Yellow  
        Log-Write -LogFile $logFile "Installing JWTDetails module: " -NoNewLine $true

        sleep 5
    
        try {
            Install-Module -Name JWTDetails -RequiredVersion 1.0.0 -Repository PSGallery -AllowClobber -Force -ErrorAction Stop
            Log-Write -LogFile $logFile "SUCCESS." -ForegroundColor "Green"
        }
        catch {
            Log-Write -LogFile $logFile "ERROR: Failed to install ImportExcel module. Script will abort." -ForegroundColor Red
            Log-Write -LogFile $logFile "Run this script 'As administrator' to install the ImportExcel module." -ForegroundColor Yellow
            Exit
        }
        Import-Module "JWTDetails" -Scope Global 
    }
    else {
        Log-Write -LogFile $logFile "JWTDetails module: " -NoNewLine $true
        Log-Write -LogFile $logFile "SUCCESS." -ForegroundColor "Green"
           
    } 
}

#######################################################################################################################
#                                               AZURE AD APP VARIABLES
#######################################################################################################################

$sufix = Get-Random 100
$appName = "CloudiwayTenantAnalysis"
$applicationName = "$appName$sufix"
# Enter the required permissions below, separated by spaces eg: "Directory.Read.All Reports.Read.All Group.ReadWrite.All Directory.ReadWrite.All"
$ApplicationPermissions = "Directory.Read.All Group.Read.All GroupMember.Read.All Sites.Read.All Team.ReadBasic.All TeamSettings.Read.All User.Read.All TeamSettings.ReadWrite.All TeamMember.Read.All TeamMember.ReadWrite.All ChannelSettings.Read.All Channel.ReadBasic.All ChannelSettings.ReadWrite.All"
# Set DelegatePermissions to $null if you only require application permissions. 
$DelegatedPermissions = $null
# Otherwise, include the required delegated permissions below.
# $DelegatedPermissions = "Directory.Read.All Group.ReadWrite.All"


#######################################################################################################################
#                                               MAIN PROGRAM
#######################################################################################################################

Write-Host 
$msg = "################################################################################################################`
                       CREATING WORKING DIRECTORIES                                                             `
################################################################################################################"
Write-Host $msg
Write-Host

#Working Directory
$workingDir = Select-Folder

#Logs directory
$logDirName = "LOGS"
$logDir = "$workingDir\$logDirName"

#Log file
$today = Get-Date -Format "yyyyMMddTHHmmss"
$logFileName = "$($today)_M365TenantAssessment.log"
$logFile = "$logDir\$logFileName"

Create-Working-Directory -workingDir $workingDir -logDir $logDir

Write-Host 
$msg = "################################################################################################################`
                       CHECKING POWERSHELL MODULES                                                             `
################################################################################################################"
Log-Write -LogFile $logFile $msg
Write-Host

Check-PSModules -LogFile $logFile

Write-Host 
$msg = "################################################################################################################`
                       CONNECTING TO MICROSOFT 365                                                             `
################################################################################################################"
Log-Write -LogFile $logFile $msg
Write-Host

Log-Write -LogFile $logFile "Enter your Microsoft 365 admin credentials."

$m365Credentials = Get-Credential -Message "Enter Your Source Microsoft 365 Admin Credentials."
if (!($m365Credentials)) {
    Log-Write -LogFile $logFile "ERROR: Cancel button or ESC was pressed while asking for Credentials. Script will abort." -ForegroundColor Red
    Exit
}
         
Connect-M365EXO -logFile $logFile -Credentials $m365Credentials 
$tenantDomain = (Get-OrganizationConfig).Name
$tenantName = $tenantDomain.replace(".onmicrosoft.com", "")
Connect-SPO  -logFile $logFile -Credentials $m365Credentials -TenantName $tenantName
Connect-AAD  -logFile $logFile -Credential $m365Credentials -TenantName $tenantName
#Graph functions
try {    
    $result =  ProcessCreationConnector  -logFile $logFile -credentials $m365Credentials -applicationName $applicationName -ApplicationPermissions $ApplicationPermissions -DelegatedPermissions $DelegatedPermissions -TenantName $tenantName
}
catch {
    Write-Host "Exception occurred while creating Azure AD App.`r`n$($_.Exception.Message)" -ForegroundColor Yellow
}

Write-Host 
$msg = "################################################################################################################`
                       EXPORTING MICROSOFT 365 TENANT OBJECTS              `
################################################################################################################"
Log-Write -LogFile $logFile $msg
Write-Host

$totalstopwatch = [System.Diagnostics.Stopwatch]::StartNew()


Analyze-M365Domains -LogFile $logFile -WorkingDir $workingDir -TenantName $tenantName 

#
#EXO PowerShell functions
Analyze-M365Mailboxes -LogFile $logFile -Credentials $m365Credentials -WorkingDir $workingDir -TenantName $tenantName 
Analyze-M365NonMailboxUsers -LogFile $logFile -WorkingDir $workingDir -TenantName $tenantName
Analyze-M365DistributionLists -LogFile $logFile -WorkingDir $workingDir -TenantName $tenantName
Analyze-M365SecurityGroups -LogFile $logFile -WorkingDir $workingDir -TenantName $tenantName -SPOUrls "https://$tenantName-admin.sharepoint.com"
Analyze-M365OrganizationalContacts -LogFile $logFile -WorkingDir $workingDir -TenantName $tenantName
#>
#Graph functions

Analyze-OneDriveForBusiness -LogFile $logFile -WorkingDir $workingDir -TenantName $tenantName -tenant_id $result.tenantId -client_id $result.clientId -client_secret $result.clientSecret 
Analyze-ClassicSPOSites -LogFile $logFile -WorkingDir $workingDir -TenantName $tenantName -tenant_id $result.tenantId -client_id $result.clientId -client_secret $result.clientSecret  
Analyze-M365UnifiedGroups -LogFile $logFile -WorkingDir $workingDir -TenantName $tenantName -tenant_id $result.tenantId -client_id $result.clientId -client_secret $result.clientSecret 
Analyze-MicrosoftTeams -LogFile $logFile -WorkingDir $workingDir -TenantName $tenantName -tenant_id $result.tenantId -client_id $result.clientId -client_secret $result.clientSecret 

Write-Host
$totalstopwatch.Stop()
Log-Write -LogFile $logFile "Total Execution time ($($totalstopwatch.Elapsed.Hours)hr:$($totalstopwatch.Elapsed.Minutes)min:$($totalstopwatch.Elapsed.Seconds)seg)" -ForegroundColor "Green"

# Removing Azure AD App
$existingapps = Get-AzureAdApplication -SearchString $appName
try {
    foreach ($existingapp in $existingapps) {
        Remove-AzureAdApplication -ObjectId $existingapp.objectId
    }    
}
catch {
    Write-Host "Exception occurred while removing Azure AD App.`r`n$($_.Exception.Message)" -ForegroundColor Red
}

Write-Host 
$msg = "################################################################################################################`
                       OPENING MICROSOFT 365 TENANT ASSESSMENT              `
################################################################################################################"
Log-Write -LogFile $logFile $msg
Write-Host

# Tenant assessment report file
$reportFilePath = "$workingDir\CloudiwayM365TenantAssessment-$tenantName.xlsx"

try {
    Start-Process -FilePath $reportFilePath
}
catch {
    $msg =    
    Log-Write -LogFile $logFile -ForegroundColor Red "ERROR: Failed to find the CSV file '$reportFilePath'." 
    Exit
}  

